<?php
/**
 * Teleport Addon
 *
 * @package NS_Cloner
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class NS_Cloner_Addon_Teleport
 *
 * Adds teleport mode to enable remote cloning between different sites.
 */
class NS_Cloner_Addon_Teleport extends NS_Cloner_Addon {

	/**
	 * Store settings for addon (e.g. connection key)
	 *
	 * @var array
	 */
	public $settings;

	/**
	 * Last error message generated by this class
	 *
	 * @var string
	 */
	public $error = '';

	/**
	 * Unique custom boundary marker for multipart encoding
	 *
	 * @var string
	 */
	private $multipart_boundary = 'bWH4JVmYCnf6GfXacrer';

	/**
	 * ID of clone mode registered by this addon.
	 */
	const MODE = 'clone_teleport';

	/**
	 * NS_Cloner_Addon_Teleport constructor.
	 */
	public function __construct() {

		$this->title       = __( 'NS Cloner Teleport', 'ns-cloner' );
		$this->plugin_path = plugin_dir_path( dirname( __FILE__ ) );
		$this->plugin_url  = plugin_dir_url( dirname( __FILE__ ) );
		parent::__construct();

		// AJAX actions that are received by the local site.
		add_action( 'wp_ajax_ns_cloner_get_remote_data', [ $this, 'ajax_get_remote_data' ] );

		// AJAX actions that are received by the remote site.
		add_action( 'wp_ajax_nopriv_nsc_verify_connection', [ $this, 'respond_to_verify_connection' ] );
		add_action( 'wp_ajax_nopriv_nsc_create_remote_site', [ $this, 'respond_to_create_remote_site' ] );
		add_action( 'wp_ajax_nopriv_nsc_process_sql', [ $this, 'respond_to_send_sql' ] );
		add_action( 'wp_ajax_nopriv_nsc_process_file', [ $this, 'respond_to_send_file' ] );
		add_action( 'wp_ajax_nopriv_nsc_process_users', [ $this, 'respond_to_send_users' ] );
		add_action( 'wp_ajax_nopriv_nsc_finish_migration', [ $this, 'respond_to_finish_migration' ] );

		// Initialize addon settings.
		$this->load_settings();

	}

	/**
	 * Runs after core modes and sections are loaded - use this to register new modes and sections
	 */
	public function init() {

		// Register new teleport mode.
		ns_cloner()->register_mode(
			self::MODE,
			[
				'title'          => __( 'Clone to Remote Site', 'ns-cloner' ),
				'button_text'    => __( 'Clone to Remote Site', 'ns-cloner' ),
				'description'    =>
                    __( 'This mode allows you to clone a site and teleport (transmit) it to another WordPress installation - even on a different server!', 'ns-cloner' ) . "\n\n" .
                    __( 'NOTE: when cloning remotely, the plugin will copy all your database content and uploads, but you\'ll need to copy your theme and plugin files manually. ', 'ns-cloner' ),
				'multisite_only' => false,
				'steps'          => [
					[ $this, 'set_up_vars' ],
					[ $this, 'copy_tables' ],
					[ $this, 'copy_files' ],
					[ $this, 'copy_users' ],
				],
				'report'         => function () {
					// Success message.
					ns_cloner()->report->add_report( '_message', __( 'Cloned to remote site successfully! ', 'ns-cloner' ) );
					// Source site.
					$source = $this->is_from_subsite() ? ns_cloner_request()->get( 'source_id' ) : null;
					ns_cloner()->report->add_report( __( 'Source Site', 'ns-cloner' ), ns_site_link( $source ) );
					// Target site.
					$remote_url  = ns_cloner_request()->get( 'target_url' );
					$remote_link = "<a href='{$remote_url}' target='_blank'>$remote_url</a>";
					ns_cloner()->report->add_report( __( 'Target Site', 'ns-cloner' ), $remote_link );
				},
			]
		);

		// Register sections.
		ns_cloner()->register_section( 'teleport_site', $this->plugin_path );
		ns_cloner()->register_section( 'teleport_target', $this->plugin_path );

		// Register background processes.
		ns_cloner()->register_process( 'teleport_tables', $this->plugin_path );
		ns_cloner()->register_process( 'teleport_rows', $this->plugin_path );
		ns_cloner()->register_process( 'teleport_files', $this->plugin_path );
		ns_cloner()->register_process( 'teleport_users', $this->plugin_path );

		// Submit finish request to remote site when cloning is finished
		// (creates users if needed, renames tables and finishes migration).
		add_action( 'ns_cloner_process_finish', [ $this, 'finish_migration' ] );

		// Clear options used to store temporary data while migrating.
		add_action( 'ns_cloner_process_exit', [ $this, 'exit_migration' ] );

		// Set up admin menu page.
		add_filter( 'ns_cloner_submenu', [ $this, 'admin_menu' ] );

		// Add option to select source section to clone full multisite network.
		add_action( 'ns_cloner_open_section_box_select_source', [ $this, 'show_full_network_option' ] );

		// Modify site tables to include/exclude user tables based on current process.
		add_filter( 'ns_cloner_site_tables', [ $this, 'site_tables_filter' ], 20 );

		// Add filter to prevent copying content of users (users process will handle that).
		add_filter( 'ns_cloner_teleport_do_copy_table', [ $this, 'do_copy_table_filter' ], 10, 2 );

		// Register names of hidden hooks that shouldn't be logged.
		add_filter( 'ns_cloner_hidden_hooks', [ $this, 'hidden_hooks_filter' ] );

		// Register keys for data that shouldn't be displayed in the logs.
		add_filter( 'ns_cloner_hidden_keys', [ $this, 'hidden_keys_filter' ] );

		// Filter process progress check to only teleport or non-teleport, to reduce unneeded queries.
		add_filter( 'ns_cloner_processes_to_check', [ $this, 'processes_filter' ] );

	}

	/**
	 * Enqueue scripts on cloner admin pages
	 */
	public function admin_enqueue() {
		wp_enqueue_script(
			'ns-cloner-teleport',
			$this->plugin_url . 'js/teleport.js',
			[ 'ns-cloner' ],
			NS_CLONER_PRO_VERSION,
			true
		);
		wp_localize_script(
			'ns-cloner-teleport',
			'ns_cloner_teleport',
			[
				'is_multisite' => is_multisite(),
			]
		);
		wp_enqueue_style(
			'ns-cloner-teleport',
			$this->plugin_url . 'css/teleport.css',
			[],
			NS_CLONER_PRO_VERSION
		);
	}

	/**
	 * Add submenu item to Cloner admin menu
	 *
	 * @param array $submenu WP submenu array.
	 * @return array
	 */
	public function admin_menu( $submenu ) {
		$submenu['ns-cloner-teleport'] = [
			ns_cloner()->menu_slug,
			__( 'Remote Connection', 'ns-cloner' ),
			__( 'Remote Connection', 'ns-cloner' ),
			ns_cloner()->capability,
			'ns-cloner-teleport',
			[ $this, 'admin_settings_page' ],
		];
		return $submenu;
	}

	/**
	 * Display the registration templates settings page
	 */
	public function admin_settings_page() {
		// Check / save settings.
		if ( ns_cloner_request()->get( 'reset_connection_key' ) && ns_cloner()->check_permissions() ) {
			$this->settings['connection_key'] = wp_generate_password( 32 );
			update_site_option( 'ns_cloner_teleport_settings', $this->settings );
		}
		// Render template.
		ns_cloner()->render( 'teleport-settings', $this->plugin_path );
	}

	/**
	 * Set settings for this mode (e.g. remote connection key), and initialize if they're blank
	 */
	public function load_settings() {
		// Try to load saved settings.
		$this->settings = get_site_option( 'ns_cloner_teleport_settings' );
		// Initialize if empty.
		if ( ! $this->settings ) {
			$this->settings = [
				'connection_key' => wp_generate_password( 32 ),
			];
			update_site_option( 'ns_cloner_teleport_settings', $this->settings );
		}
	}

	/**
	 * Add radio buttons for full network cloning at beginning of 'Select Source' section
	 */
	public function show_full_network_option() {
		if ( is_multisite() && is_network_admin() ) :
			?>
			<div class="ns-cloner-teleport-network-selector">
				<input type="radio" name="teleport_full_network" id="full_network_0" value="" checked/>
				<label for="full_network_0"><?php esc_html_e( 'Select a single site to clone', 'ns-cloner' ); ?></label>
				<input type="radio" name="teleport_full_network" id="full_network_1" value="1"/>
				<label for="full_network_1"><?php esc_html_e( 'Or clone entire multisite network', 'ns-cloner' ); ?></label>
			</div>
			<?php
		endif;
	}

	/**
	 * Adjust list of site tables
	 *
	 * Includes user tables for most teleports, removes them when cloning to a subsite
	 * to prevent deleting other network users.
	 *
	 * @param array $tables List of site database tables.
	 * @return array
	 */
	public function site_tables_filter( $tables ) {
		if ( ns_cloner_request()->is_mode( 'clone_teleport' ) ) {
			$base_prefix = ns_cloner()->db->base_prefix;
			$user_tables = [ "{$base_prefix}users", "{$base_prefix}usermeta" ];
			if ( $this->is_to_subsite() ) {
				// Don't clone users + usermeta tables if cloning to subsite (would delete other network users).
				ns_cloner()->log->log( 'REMOVING users and usermeta tables for clone to subsite.' );
				$tables = array_diff( $tables, $user_tables );
			} else {
				// Do clone users and usermeta tables for other clones.
				ns_cloner()->log->log( 'ADDING users and usermeta tables for remote clone.' );
				$tables = array_unique( array_merge( $tables, $user_tables ) );
			}
		}
		return $tables;
	}

	/**
	 * Register filter to prevent copy content of user and usermeta tables.
	 *
	 * We need the user tables to be dropped and recreated, but no
	 * rows to be copied because the teleport users process will handle
	 * cloning the users and meta in a single/multi compatible way.
	 *
	 * @param bool   $do_copy Whether to copy the row.
	 * @param string $table Source table name.
	 * @return bool
	 */
	public function do_copy_table_filter( $do_copy, $table ) {
		if ( preg_match( '/(users|usermeta)$/', $table ) ) {
			$do_copy = false;
		}
		return $do_copy;
	}

	/**
	 * Register names of hooks that shouldn't be logged
	 *
	 * @param array $hooks List of hooks not to log.
	 * @return array
	 */
	public function hidden_hooks_filter( $hooks ) {
		array_push( $hooks, 'ns_cloner_teleport_request' );
		return $hooks;
	}

	/**
	 * Register keys for data that shouldn't appear in the logs
	 *
	 * @param array $keys Array of keys not to display.
	 * @return array
	 */
	public function hidden_keys_filter( $keys ) {
		array_push( $keys, 'remote_key' );
		array_push( $keys, 'fragment' );
		return $keys;
	}

	/**
	 * Filter process progress check to only teleport or non-teleport, to reduce unneeded queries.
	 *
	 * @param array $processes Array of registered NS_Cloner_Process objects.
	 * @return array
	 */
	public function processes_filter( $processes ) {
		foreach ( $processes as $id => $process ) {
			$is_teleport_mode    = ns_cloner_request()->is_mode( 'clone_teleport' );
			$is_teleport_process = preg_match( '/^teleport/', $id );
			if ( ( $is_teleport_mode && ! $is_teleport_process ) || ( $is_teleport_process && ! $is_teleport_mode ) ) {
				unset( $processes[ $id ] );
			}
		}
		return $processes;
	}

	/*
	_____________________________________
	|
	|  Clone Steps
	|_____________________________________
	*/

	/**
	 * Get variables needed for cloning upload dir and db prefix from the remote site
	 *
	 * This is similar to NS_Cloner_Request->set_up_vars() but handles all the differences
	 * required to work with remote cloning, rather than setting up target vars for a local subsite.
	 * This includes creating a new subsite to get variables from, IF the remote site is multisite
	 * AND the full network option is not selected (otherwise it would just overwrite either the
	 * whole remote single site or the whole remote multisite).
	 */
	public function set_up_vars() {
		// Set up source/local vars. This will be handled by default for cloning a single multisite blog,
		// but it needs to be done manually if the source site is full-network or non-multisite.
		$source_id   = ns_cloner_request()->get( 'source_id' );
		$source_vars = ns_cloner_request()->define_vars( $source_id );
		foreach ( $source_vars as $key => $value ) {
			ns_cloner_request()->set( "source_{$key}", $value );
		}

		// If a single site or single blog on a multisite is being cloned to a remote multisite,
		// we need to create the remote sub-site, and use its newly generated variables as target vars.
		if ( $this->is_to_subsite() ) {
			$target_vars = $this->create_remote_site();
			if ( $target_vars ) {
				$this->set_remote_data( [ 'target_vars' => $target_vars ] );
			} else {
			    return false;
			}
		}

		// Set up target variables from the remote site.
		$target_vars = $this->get_remote_data( 'target_vars' );
		foreach ( $target_vars as $key => $value ) {
			ns_cloner_request()->set( "target_{$key}", $value );
		}

		// Save search/replace data so it will be available for background processes.
		ns_cloner_request()->set_up_search_replace( $source_id, 'remote' );

		// Add finish query to rename site, if applicable, since it will get overwritten by table cloning.
		$target_title = ns_cloner_request()->get( 'teleport_target_title' );
		if ( $target_title ) {
			$key = $this->is_full_network() ? 'site_name' : 'blogname';
			ns_cloner()->process_manager->add_finish_query( "option::$key::$target_title" );
		}

		// Save the updated request for later access.
		ns_cloner_request()->save();
	}

	/**
	 * Queue tables for migration.
	 *
	 * This is the first step for the teleport mode.
	 */
	public function copy_tables() {
		$tables_process = ns_cloner()->get_process( 'teleport_tables' );
		$source_id      = ns_cloner_request()->get( 'source_id' );
		$source_tables  = ns_cloner()->get_site_tables( $source_id );

		// Queue tables to background process.
		foreach ( ns_reorder_tables( $source_tables ) as $source_table ) {
			$table_data = [
				'source_id'    => $source_id,
				'source_table' => $source_table,
				'target_table' => $this->temp_table_name( $source_table ),
			];
			$tables_process->push_to_queue( $table_data );
		}

		$tables_process->save()->dispatch();
	}

	/**
	 * Queue files for migration.
	 *
	 * This is the second step for the teleport mode.
	 */
	public function copy_files() {
		// Makes sure file copy is not unchecked.
		if ( ! ns_cloner_request()->get( 'do_copy_files' ) ) {
			ns_cloner()->log->log( 'SKIPPING teleport copy_files step because *do_copy_files* was false' );
			return;
		}

		// For full network copy, don't ignore 'sites' dir like ns_recursive_dir_copy_by_process usually does.
		if ( $this->is_full_network() ) {
			add_filter(
				'ns_cloner_dir_copy_ignore',
				function ( $to_ignore ) {
					return array_diff( $to_ignore, [ 'sites' ] );
				}
			);
		}

		// Recursively queue files.
		$source_dir    = ns_cloner_request()->get( 'source_upload_dir' );
		$target_dir    = ns_cloner_request()->get( 'target_upload_dir' );
		$files_process = ns_cloner()->get_process( 'teleport_files' );
		$num_files     = ns_recursive_dir_copy_by_process( $source_dir, $target_dir, $files_process );
		ns_cloner()->log->log( "QUEUEING *$num_files* files from *$source_dir* to *$target_dir* (remote)" );

		$files_process->save()->dispatch();
	}

	/**
	 * Queue existing users for migration.
	 *
	 * New users will be added at the end of the process (not here or by the background process)
	 * so that they can take advantage of the default WP field and meta population for new users.
	 *
	 * This is the third step for the teleport mode.
	 */
	public function copy_users() {
		$source_id     = ns_cloner_request()->get( 'source_id' );
		$source_prefix = ns_cloner_request()->get( 'source_prefix' );
		$target_prefix = ns_cloner_request()->get( 'target_prefix' );
		$users_process = ns_cloner()->get_process( 'teleport_users' );

		// Determine which users to copy.
		if ( ! ns_cloner_request()->get( 'do_copy_users' ) ) {
			$users = [];
		} else {
			// If do_copy_users is true, get the appropriate list.
			if ( $this->is_from_subsite() ) {
				// For cloning from subsite, add source blog's users only (not other network users).
				$users = get_users( "blog_id={$source_id}&exclude=" . get_current_user_id() );
			} else {
				// Otherwise copy all users.
				$users = get_users( 'blog_id=0&exclude=' . get_current_user_id() );
			}
		}
		// Always include the current user.
		array_unshift( $users, wp_get_current_user() );

		// Load both the user itself and its usermeta entries into the process queue.
		foreach ( $users as $user ) {
			$users_process->push_record_to_queue( 'users', (array) $user->data );
			$user_meta = ns_cloner()->db->get_results(
				ns_cloner()->db->prepare(
					'SELECT * FROM ' . ns_cloner()->db->usermeta . ' WHERE user_id = %d',
					$user->ID
				),
				ARRAY_A
			);
			foreach ( $user_meta as $meta_row ) {
				if ( preg_match( "/$source_prefix(user_level|capabilities)/", $meta_row['meta_key'], $matches ) ) {
					// Update user level / capability prefixes (e.g. change wp_3_capabilities to remote_4_capabilities).
					$meta_row['meta_key'] = $target_prefix . $matches[1];
				} elseif ( preg_match( "/$source_prefix(\d+_)?(user_level|capabilities)/", $meta_row['meta_key'] ) ) {
					if ( ! $this->is_full_network() ) {
						// Skip levels/capabilities for sites other than the source site (for non-full network clones).
						continue;
					} else {
						// Do include levels/caps for all subsites when cloning full network.
						$meta_row['meta_key'] = preg_replace( "/^$source_prefix/", $target_prefix, $meta_row['meta_key'] );
					}
				}
				$users_process->push_record_to_queue( 'usermeta', $meta_row, true );
			}
		}

		// Save users queue, only dispatch now if there are no tables. Otherwise will be auto-dispatched when tables are done.
		if ( ns_cloner()->get_process( 'teleport_tables' )->is_queue_empty() ) {
			$users_process->save()->dispatch();
		} else {
			$users_process->save();
		}
	}

	/*
	_______________________________________
	|
	|  Verify Connection
	|______________________________________
	*/

	/**
	 * Test the connection to a remote teleport instance, and save remote site data
	 *
	 * This will be run automatically by the validation for the teleport section
	 * before the cloning process begins.
	 *
	 * @return bool|array
	 */
	public function verify_connection() {
		$data   = [
			'action'         => 'nsc_verify_connection',
			'plugin_version' => NS_CLONER_PRO_VERSION,
		];
		$result = $this->request_to_remote_site( $data );

		if ( ! $result ) {
			return false;
		} else {
			// Successful verification. Save remote site data for later use.
			$this->set_remote_data( $result );
			return $result;
		}
	}

	/**
	 * Respond to verification request
	 */
	public function respond_to_verify_connection() {
		global $wp_version, $wp_db_version;

		// Prepare request data and check signature.
		$data = $this->receive_request( [ 'plugin_version' ] );

		// Check remote plugin version compatibility.
		$source_version = $data['plugin_version'];
		$target_version = NS_CLONER_PRO_VERSION;
		if ( version_compare( $source_version, $target_version, '!=' ) ) {
			$version_error = sprintf(
				'<b>Version Mismatch</b> &mdash; It appears you have NS Cloner Pro %1$s on %2$s, but are using %3$s here. Please update to use the same version.',
				$target_version,
				ns_short_url( site_url() ),
				$source_version
			);
			wp_send_json_error( $version_error );
		}

		// Successful verification - return site data.
		$site_data = [
			'gzip_supported'     => function_exists( 'gzuncompress' ),
			'post_max_size'      => $this->get_post_max_size(),
			'max_allowed_packet' => $this->get_max_allowed_packet(),
			'is_multisite'       => is_multisite(),
			'is_subdomain'       => is_multisite() && is_subdomain_install(),
			'wp_version'         => $wp_version,
			'wp_db_version'      => $wp_db_version,
			'base_prefix'        => ns_cloner()->db->base_prefix,
			'target_vars'        => ns_cloner_request()->define_vars(),
		];
		wp_send_json_success( $site_data );
	}

	/*
	_______________________________________
	|
	|  Create Remote Site
	|______________________________________
	*/

	/**
	 * Create a new site/blog on a remote network
	 */
	public function create_remote_site() {
		// Send site creation request to remote multisite.
		$data   = [
			'action' => 'nsc_create_remote_site',
			'title'  => ns_cloner_request()->get( 'teleport_target_title' ),
			'name'   => strtolower( trim( ns_cloner_request()->get( 'teleport_target_name' ) ) ),
		];
		$result = $this->request_to_remote_site( $data );

		// Handle results.
		if ( ! $result ) {
			$error_message = __( 'Error creating remote site.', 'ns-cloner' ) . ' ' . $this->error;
			ns_cloner()->process_manager->exit_processes( $error_message );
		} else {
			ns_cloner()->log->log( 'New remote site created: ' . $result['url'] );
		}

		return $result;
	}

	/**
	 * Runs on remote site - respond to create_remote_site() above
	 *
	 * We can safely use multisite functions here because this will only be called for multisite cloning.
	 */
	public function respond_to_create_remote_site() {
		// Prepare request data and check signature.
		$data = $this->receive_request( [ 'title', 'name' ] );

		// Validate it.
		$validation_errors = ns_wp_validate_site( $data['name'], $data['title'] );
		if ( ! empty( $validation_errors ) ) {
			wp_send_json_error( $validation_errors[0] );
		}

		// Set up new site data. Based on calculated values in wpmu_validate_blog_signup().
		$site_data = [
			'title' => $data['title'],
			// User id will be updated at the end, after users process is done.
		];
		if ( is_subdomain_install() ) {
			$site_data += [
				'domain' => $data['name'] . '.' . preg_replace( '|^www\.|', '', get_current_site()->domain ),
				'path'   => get_current_site()->path,
			];
		} else {
			$site_data += [
				'domain' => get_current_site()->domain,
				'path'   => get_current_site()->path . $data['name'] . '/',
			];
		}

		// Create new blog/site for source site to be cloned onto, and handle results.
		$target_id = wp_insert_site( $site_data );
		if ( is_wp_error( $target_id ) ) {
			wp_send_json_error( $target_id->get_error_message() );
		} else {
			wp_send_json_success( ns_cloner_request()->define_vars( $target_id ) );
		}
	}

	/*
	_______________________________________
	|
	|  File Transfer
	|______________________________________
	*/

	/**
	 * Collect and transfer file
	 *
	 * @param string $source Current file path on source site.
	 * @param string $destination Desired file path on destination site.
	 * @return bool
	 */
	public function send_file( $source, $destination ) {
		ns_cloner()->log->log( "ENTERING *send_file* for file *$source* and destination *$destination*." );

		// Make sure that file exists.
		if ( ! file_exists( $source ) ) {
			ns_cloner()->report->add_notice( 'File ' . $source . ' was skipped because it could not be found on the server.' );
			ns_cloner()->log->log( 'File not found. Skipping.' );
			return true;
		}

		// Make sure file is not too big.
		$file_size  = filesize( $source );
		$file_limit = $this->get_remote_data( 'post_max_size' );
		if ( $file_size > $file_limit ) {
			ns_cloner()->report->add_notice( 'File ' . $source . ' was skipped because it exceeds the server\'s post size limit' );
			ns_cloner()->log->log( 'File is too big (' . size_format( $file_size ) . '). Skipping.' );
			return true;
		}

		// Send to remote site.
		$file     = file_get_contents( $source );
		$file     = base64_encode( $file );
		$data     = [
			'action'      => 'nsc_process_file',
			'destination' => $destination,
			'gzipped'     => $this->is_gzip_supported() ? '1' : '0',
			'fragment'    => $this->is_gzip_supported() ? gzcompress( $file ) : $file,
		];
		$response = $this->request_to_remote_site( $data );

		// File transfer error should be non fatal to the overall process -
		// we don't want one htaccess file or similar to fail the whole thing.
		// So add notice, but always return true.
		if ( ! $response ) {
			ns_cloner()->report->add_notice( $this->error );
			$this->error = '';
		}
		return true;
	}

	/**
	 * Process file on remove host
	 */
	public function respond_to_send_file() {
		// Prepare request data and check signature.
		$data = $this->receive_request( [ 'destination', 'gzipped', 'fragment' ] );

		// Create parent directory if it doesn't yet exist.
		$destination       = $data['destination'];
		$destination_array = explode( '/', $destination );
		$file_name         = array_pop( $destination_array );
		$dir               = str_replace( $file_name, '', $destination );
		if ( ! is_dir( $dir ) ) {
			mkdir( $dir, 0777, true );
		}

		// Try saving the file and send the results back.
		$file   = base64_decode( $data['fragment'] );
		$result = file_put_contents( $destination, $file );
		if ( false === $result ) {
			/* translators: file path */
			$error = sprintf( __( 'File %1$s could not be saved on the remote site.', 'ns-cloner' ), $destination );
			wp_send_json_error( $error );
		} else {
			wp_send_json_success();
		}

	}

	/*
	_______________________________________
	|
	|  Table transfer
	|______________________________________
	*/

	/**
	 * Send a 'fragment' or collection of SQL from a table to the remote site
	 *
	 * @param string $sql String of queries to be executed.
	 * @return bool|mixed
	 */
	public function send_sql( $sql ) {
		ns_cloner()->log->log( [ 'ENTERING *send_sql* with query:', $sql ] );

		$data     = [
			'action'   => 'nsc_process_sql',
			'gzipped'  => $this->is_gzip_supported() ? '1' : '0',
			'fragment' => $this->is_gzip_supported() ? gzcompress( $sql ) : $sql,
		];
		$response = $this->request_to_remote_site( $data );

		return $response;
	}

	/**
	 * Receive a fragment on the remote site and save to db
	 */
	public function respond_to_send_sql() {
		// Prepare request data and check signature.
		$data = $this->receive_request( [ 'gzipped', 'fragment' ] );

		// Loop through all queries and process, breaking on error.
		$queries = array_filter( explode( ";\n", $data['fragment'] ) );
		foreach ( $queries as $query ) {
			ns_cloner()->db->query( $query );
			$this->handle_remote_db_error();
		}

		wp_send_json_success();
	}

	/*
	_______________________________________
	|
	|  Users transfer
	|______________________________________
	*/

	/**
	 * Send a collection of users and user meta to the remote site
	 *
	 * @param array $users Array of records to be transferred.
	 * @return bool|mixed
	 */
	public function send_users( $users ) {
		ns_cloner()->log->log( [ 'ENTERING *send_users* with payload:', $users ] );
		$fragment = wp_json_encode( $users );

		$data     = [
			'action'        => 'nsc_process_users',
			'is_to_subsite' => $this->is_to_subsite() ? '1' : '0',
			'gzipped'       => $this->is_gzip_supported() ? '1' : '0',
			'fragment'      => $this->is_gzip_supported() ? gzcompress( $fragment ) : $fragment,
		];
		$response = $this->request_to_remote_site( $data );

		return $response;
	}

	/**
	 * Receive a fragment of user data on the remote site and save to db
	 */
	public function respond_to_send_users() {
		$data = $this->receive_request( [ 'action', 'is_to_subsite', 'gzipped', 'fragment' ] );

		// Loop through all users/meta and process, using flexible primary keys.
		$changes = get_site_option( 'ns_cloner_teleport_user_id_changes', [] );
		$items   = json_decode( $data['fragment'], true );
		foreach ( $items as $item ) {
			$record = $item['record'];
			$table  = $item['table'];
			// Remove multisite user columns if on single site.
			if ( ! is_multisite() && 'users' === $table ) {
				$record = array_diff_key( $record, array_flip( [ 'spam', 'deleted' ] ) );
			}
			// Handle insertion.
			if ( '1' !== $data['is_to_subsite'] ) {
				// If we're inserting into a blank temp table, we can safely just insert with existing primary ID's.
				$prefix = ns_cloner()->temp_prefix . ns_cloner()->db->base_prefix;
				ns_cloner()->db->insert( $prefix . $table, $record );
			} else {
				// If we're inserting into an existing multisite table, we'll have to replace primary ID's.
				// Check that user doesn't already exist before inserting.
				if ( 'users' === $table ) {
					// If user exists, skip this user and map it to false, so associated user meta will get skipped too.
					$old_user_id  = $record['ID'];
					$email_exists = get_user_by( 'email', $record['user_email'] );
					$login_exists = get_user_by( 'login', $record['user_login'] );
					if ( $email_exists || $login_exists ) {
						$changes[ $old_user_id ] = false;
						continue;
					}
					// Clear primary key to avoid conflicts.
					unset( $record['ID'] );
				}
				// Update user ID for user meta, if needed.
				if ( 'usermeta' === $table ) {
					$old_user_id = $record['user_id'];
					if ( isset( $changes[ $old_user_id ] ) ) {
						$new_user_id = $changes[ $old_user_id ];
						if ( ! $new_user_id ) {
							// Skip this whole row if no new primary key val.
							continue;
						} else {
							// Otherwise update the primary key value and keep going.
							$record['user_id'] = $new_user_id;
						}
					}
					// Clear primary key to avoid conflicts.
					unset( $record['umeta_id'] );
				}
				// Insert the row into the live table (no temp user tables for cloning to subsite).
				ns_cloner()->db->insert( ns_cloner()->db->base_prefix . $table, $record );
				// Update user id changelist, so usermeta rows to come can know what their user id changed to.
				if ( 'users' === $table && ns_cloner()->db->insert_id ) {
					$new_user_id             = ns_cloner()->db->insert_id;
					$changes[ $old_user_id ] = $new_user_id;
					// Save query for updating post/comment authors.
					$prefix = ns_cloner()->temp_prefix . ns_cloner()->db->base_prefix;
					ns_cloner()->process_manager->add_finish_query(
						ns_cloner()->db->prepare(
							"UPDATE {$prefix}posts SET post_author=%d WHERE post_author=%d",
							$new_user_id,
							$old_user_id
						)
					);
					ns_cloner()->process_manager->add_finish_query(
						ns_cloner()->db->prepare(
							"UPDATE {$prefix}comments SET user_id=%d WHERE user_id=%d",
							$new_user_id,
							$old_user_id
						)
					);
					do_action( 'ns_cloner_teleport_change_user_id', $new_user_id, $old_user_id );
				}
			}
		}
		update_site_option( 'ns_cloner_teleport_user_id_changes', $changes );

		wp_send_json_success();
	}

	/*
	_______________________________________
	|
	|  Finish Migration
	|______________________________________
	*/

	/**
	 * Send request to remote site to rename temporary migrated tables once all data is cloned.
	 *
	 * Runs on ns_cloner_process_finish action.
	 */
	public function finish_migration() {
		if ( ns_cloner_request()->is_mode( self::MODE ) ) {
			// Prepare list of temporary tables that should be renamed.
			$source_id   = ns_cloner_request()->get( 'source_id' );
			$tables      = ns_cloner()->get_site_tables( $source_id );
			$temp_tables = array_map( [ $this, 'temp_table_name' ], $tables );

			// Add finish query to set teleport remote key back to original value so it doesn't change after clone.
			$teleport_settings = serialize( [ 'connection_key' => ns_cloner_request()->get( 'remote_key' ) ] );
			ns_cloner()->process_manager->add_finish_query( 'option::ns_cloner_teleport_settings::' . $teleport_settings );

			// Get copy of finish queries for sending to remote site,
			// then delete them from the db so they won't be run on the local site by the process manager.
			$finish_queries = ns_cloner()->process_manager->get_finish_queries();
			delete_site_option( 'ns_cloner_finish_queries' );

			// Compile user data for any new users specified.
			$new_user_names  = ns_cloner_request()->get( 'new_user_names', [] );
			$new_user_emails = ns_cloner_request()->get( 'new_user_emails', [] );
			$new_user_pairs  = array_combine( $new_user_names, $new_user_emails );
			$do_user_notify  = ns_cloner_request()->get( 'do_user_notify', false );

			// Submit request to remote site to run last actions that have to be run at the end of the migration.
			$data     = [
				'action'         => 'nsc_finish_migration',
				'tables'         => implode( ',', $temp_tables ),
				'finish_queries' => implode( '||', $finish_queries ),
				'users'          => wp_json_encode( $new_user_pairs ),
				'do_user_notify' => $do_user_notify,
				'target_id'      => ns_cloner_request()->get( 'target_id' ),
			];
			$response = $this->request_to_remote_site( $data );
			// Handle any errors.
			if ( ! $response ) {
				ns_cloner()->process_manager->exit_processes( $response['error'] );
			}
		}
	}

	/**
	 * Runs on remote site - respond to finish_migration remote request above
	 */
	public function respond_to_finish_migration() {
		$data = $this->receive_request( [ 'tables', 'finish_queries', 'users', 'do_user_notify', 'target_id' ] );

		// Move temporary migration tables to final location.
		$tables = array_filter( explode( ',', $data['tables'] ) );
		foreach ( $tables as $temp_table ) {
			// Make sure replacement table exists before dropping existing one.
			$table_exists = ns_cloner()->db->get_var( "SHOW TABLES LIKE '" . ns_cloner()->db->esc_like( $temp_table ) . "'" );
			if ( $table_exists ) {
				$final_table = str_replace( ns_cloner()->temp_prefix, '', $temp_table );
				// Drop existing remote table.
				$drop_query = 'DROP TABLE IF EXISTS ' . ns_sql_backquote( $final_table ) . ';';
				ns_cloner()->db->query( $drop_query );
				$this->handle_remote_db_error();
				// Rename migration table to its real name.
				$rename_query = 'RENAME TABLE ' . ns_sql_backquote( $temp_table ) . ' TO ' . ns_sql_backquote( $final_table ) . ';';
				ns_cloner()->db->query( $rename_query );
				$this->handle_remote_db_error();
			} else {
				/* translators: database table name */
				$error = sprintf( __( 'Missing table %s on remote site. Could not complete migration.', 'ns-cloner' ), $temp_table );
				wp_send_json_error( $error );
			}
		}

		// Run finish queries (eg alter table statements for constraints).
		$finish_queries = array_filter( explode( '||', $data['finish_queries'] ) );
		foreach ( $finish_queries as $query ) {
			if ( preg_match( '/^option::(.+)::(.+)$/', $query, $parts ) ) {
				// Handle option-type shortcut references in the format 'option::KEY::VALUE'.
				ns_cloner()->db->query(
					ns_prepare_option_query(
						'UPDATE {table} SET {value} = %s WHERE {key} = %s',
						[ $parts[2], $parts[1] ]
					)
				);
			} else {
				// Handle SQL statements.
				ns_cloner()->db->query( $query );
				$this->handle_remote_db_error();
			}
		}

		// Create new users (cannot rely on any multisite functions here because this could be single site).
		$new_user_ids = [];
		$new_users    = json_decode( $data['users'], true );
		// Disable user notifications if requested.
		if ( ! $data['do_user_notify'] ) {
			remove_action( 'register_new_user', 'wp_send_new_user_notifications' );
		}
		foreach ( $new_users as $login => $email ) {
			// This will just have to fail quietly if it doesn't work, because it would be crazy to return
			// an error here after the user has waited through the whole cloning process for such a non-critical
			// item (it's easy to manually add the new user if it doesnt work automatically).
			$new_id = register_new_user( $login, $email );
			if ( ! is_wp_error( $new_id ) ) {
				get_user_by( 'id', $new_id )->set_role( 'administrator' );
				$new_user_ids[] = $new_id;
			}
		}

		// For multisite, need to add new users as well as any previous super admins to the new cloned site.
		if ( is_multisite() && is_numeric( $data['target_id'] ) ) {
			$users_to_add = array_merge( $new_user_ids, get_super_admins() );
			foreach ( $users_to_add as $user_id ) {
				add_user_to_blog( $data['target_id'], $user_id, 'administrator' );
			}
		}

		wp_send_json_success();
	}

	/**
	 * Clear saved options data for the current teleport cloning process.
	 *
	 * Runs on ns_cloner_process_exit action.
	 */
	public function exit_migration() {
		delete_site_option( 'ns_cloner_teleport_user_id_changes' );
		delete_site_option( 'ns_cloner_teleport_remote_data' );
	}

	/*
	_______________________________________
	|
	|  Remote Request / Signature utilities
	|______________________________________
	*/

	/**
	 * Sign data with an encryption key for remote authentication
	 *
	 * @param mixed  $data Data to generate signature from.
	 * @param string $key Key to sign with.
	 * @return string
	 */
	private function generate_signature( $data, $key ) {
		// Remove signature if present, since that would be circular and never validate.
		if ( isset( $data['signature'] ) ) {
			unset( $data['signature'] );
		}
		// Convert booleans to strings.
		foreach ( $data as $i => $v ) {
			if ( is_bool( $v ) ) {
				$data[ $i ] = $v ? 'true' : 'false';
			}
		}
		$single_string = implode( '', $data );
		return base64_encode( hash_hmac( 'md5', $single_string, $key, true ) );
	}

	/**
	 * Sign incoming data and compare signatures to determine if request is authorized.
	 *
	 * @param array $data Data to generate signature from.
	 * @return bool
	 */
	private function check_signature( $data ) {
		if ( empty( $data['signature'] ) ) {
			return false;
		}
		$connection_key = $this->settings['connection_key'];
		$new_signature  = $this->generate_signature( $data, $connection_key );
		return $new_signature === $data['signature'];
	}

	/**
	 * Send request to remote teleport instance
	 *
	 * This returns the response data on success, or false on failure.
	 * In the case of failure, it will populate $this->error with the error message.
	 *
	 * @param array $data Request data.
	 * @return bool|mixed
	 */
	public function request_to_remote_site( $data ) {
		// Remove timeout, if allowed.
		$this->set_time_limit();

		// Add signature to data, for remote site to check.
		$data['signature'] = $this->generate_signature( $data, ns_cloner_request()->get( 'remote_key' ) );

		// Make WP ajax request to remote teleport.
		$url  = trailingslashit( ns_cloner_request()->get( 'remote_url' ) ) . 'wp-admin/admin-ajax.php';
		$args = apply_filters(
			'ns_cloner_teleport_request',
			[
				'body'     => $this->array_to_multipart( $data ),
				'blocking' => true,
				'timeout'  => 60,
				'headers'  => [
					'Content-Type'     => 'multipart/form-data; boundary=' . $this->multipart_boundary,
					'Referer'          => $url,
					'Content-Encoding' => 'gzip',
				],
			]
		);
		ns_cloner()->log->log( [ 'SENDING teleport request:', $data ] );
		$request = wp_remote_post( $url, $args );

		// Prepare data from response.
		$message = wp_remote_retrieve_response_message( $request );
		$code    = wp_remote_retrieve_response_code( $request );
		$body    = trim( wp_remote_retrieve_body( $request ), "\xef\xbb\xbf" ); // Strip BOM.
		$result  = is_null( json_decode( $body ) ) ? $body : json_decode( $body, true );

		// Handle any errors.
		if ( is_wp_error( $request ) ) {
			// Handle invalid response.
			$this->error = sprintf(
				'Connection to %s failed. %s',
				ns_cloner_request()->get( 'remote_url' ),
				$request->get_error_message()
			);
		} elseif ( '0' === $result ) {
			// Handle 0 - means that WP ajax is running, but nothing hooked to the teleport action.
			$this->error = sprintf( 'It appears that NS Cloner Pro isn\'t installed on %s', ns_cloner_request()->get( 'remote_url' ) );
		} elseif ( $code < 200 || $code > 399 ) {
			// Handle valid response with HTTP error code present.
			$this->error = sprintf( 'Request to remote site failed with HTTP code %1$s - %2$s', $code, $message );
		} elseif ( isset( $result['success'] ) && ! $result['success'] ) {
			// Handle responses sent with wp_json_send_error().
			$this->error = isset( $result['data'] ) ? $result['data'] : 'Invalid or empty response from remote site.';
		}

		// Return false if error occurred, or request body if successful.
		if ( ! empty( $this->error ) ) {
			// Just false for simple processing. Error message is accessible via $this->error.
			ns_cloner()->log->log( [ 'RECEIVED invalid teleport response:', $request ] );
			return false;
		} else {
			// For wp_send_json_success responses, just return the useful portion - data.
			$data = is_array( $result ) && isset( $result['data'] ) ? $result['data'] : $result;
			ns_cloner()->log->log( [ 'RECEIVED teleport response content:', $data ] );
			return $data;
		}
	}

	/**
	 * Receive request on remote site, and do basic setup and signature checking.
	 *
	 * This is the counterpart to request_to_remote_site(). It does all the basic cleaning processing,
	 * and setup so that the different respond ajax hook functions don't have to repeat this.
	 *
	 * @param array $keys Array of accepted keys that should be present in received request.
	 * @return array
	 */
	private function receive_request( $keys ) {
		// Get only POST elements for the accepted keys (and always include action and signature).
		$keys         = array_merge( $keys, [ 'signature', 'action' ] );
		$flipped_keys = array_flip( $keys );
		$post         = array_intersect_key( $_POST, $flipped_keys );
		$data         = wp_unslash( $post );
		$gzipped      = ( isset( $data['gzipped'] ) && '1' === $data['gzipped'] );

		// Decode fragment if there should be one (sent as multipart file, so can't get it from $_POST).
		if ( isset( $flipped_keys['fragment'] ) ) {
			$data['fragment'] = $this->get_fragment( $gzipped );
		}

		// Check for valid signature.
		if ( ! $this->check_signature( $data ) ) {
			wp_send_json_error( __( 'Remote authentication failed. Please check that you have the correct remote key.', 'ns-cloner' ) );
		}

		// Unzip should be only after check the signature.
		if ( isset( $data['fragment'] ) ) {
			$data['fragment'] = $gzipped ? gzuncompress( $data['fragment'] ) : $data['fragment'];
		}

		// Prepare for handling - try to clear time limit, and hide db errors so they won't be dumped in JSON.
		$this->set_time_limit();
		ns_cloner()->db->hide_errors();

		return $data;
	}

	/*
	_______________________________________
	|
	|  Info / General utilities
	|______________________________________
	*/

	/**
	 * Save/store data about the remote site
	 *
	 * This is the data that gets returned when verifying the connection.
	 * It is also updated when creating a new remote sub-site.
	 *
	 * @param array $data Associative array of data to save.
	 */
	public function set_remote_data( $data ) {
		$saved   = $this->get_remote_data();
		$updated = array_merge( $saved, $data );
		update_site_option( 'ns_cloner_teleport_remote_data', $updated );
		ns_cloner()->log->log( [ 'UPDATING teleport remote data', $updated ] );
	}

	/**
	 * Get stored data about the remote site
	 *
	 * @param string|null $key Optional key to retrieve value for.
	 * @return string|array
	 */
	public function get_remote_data( $key = null ) {
		$data = get_site_option( 'ns_cloner_teleport_remote_data', [] );
		if ( ! is_null( $key ) ) {
			return isset( $data[ $key ] ) ? $data[ $key ] : '';
		} else {
			return $data;
		}
	}

	/**
	 * Get remote data by AJAX
	 */
	public function ajax_get_remote_data() {
		ns_cloner()->check_permissions();
		wp_send_json_success( $this->get_remote_data() );
	}

	/**
	 * Try to set unlimited execution time limit (before executing data transfer)
	 */
	private function set_time_limit() {
		if ( ! function_exists( 'ini_get' ) || ! ini_get( 'safe_mode' ) ) {
			set_time_limit( 0 );
		}
	}

	/**
	 * Check if the full network is being cloned or not (just a single site)
	 *
	 * @return bool
	 */
	public function is_full_network() {
		return ns_cloner_request()->get( 'teleport_full_network' );
	}

	/**
	 * Check if we're cloning a single subsite from a network
	 *
	 * @return bool
	 */
	public function is_from_subsite() {
		return is_multisite() && ! $this->is_full_network();
	}

	/**
	 * Check if we're cloning TO a single subsite on another network
	 *
	 * @return bool
	 */
	public function is_to_subsite() {
		return $this->get_remote_data( 'is_multisite' ) && ! $this->is_full_network();
	}

	/*
	_______________________________________
	|
	|  SQL utilities
	|______________________________________
	*/

	/**
	 * Change source table name to use temporary target prefix
	 *
	 * @param string $source_table Table name of source table.
	 * @return string
	 */
	public function temp_table_name( $source_table ) {
		$base_prefix   = ns_cloner()->db->base_prefix;
		$source_prefix = ns_cloner_request()->get( 'source_prefix' );
		$target_prefix = ns_cloner_request()->get( 'target_prefix' );
		// Include match for base prefix, because we could have source prefix like wp_2 but be including
		// wp_users tables which doesn't have the numeric prefix, and thus wouldn't be matched.
		$target_table = preg_replace( "/^($source_prefix|$base_prefix)/", $target_prefix, $source_table );
		return ns_cloner()->temp_prefix . $target_table;
	}

	/**
	 * Run after every SQL query on the remote site, to break operation and send error if one occurred
	 */
	private function handle_remote_db_error() {
		$error = ns_cloner()->db->last_error;
		if ( ! empty( $error ) ) {
			wp_send_json_error( __( 'SQL error on remote site.', 'ns-cloner' ) . ' ' . $error );
		}
	}

	/**
	 * Get the maximum SQL packet size (query length)
	 *
	 * @return int
	 */
	private function get_max_allowed_packet() {
		$default  = 50000;
		$variable = ns_cloner()->db->get_var( "SHOW VARIABLES LIKE 'max_allowed_packet'" );
		return 0 === (int) $variable ? $default : $variable;
	}

	/*
	_______________________________________
	|
	|  File / Data Utilities
	|______________________________________
	*/

	/**
	 * Check if gzip compression is available for both local and remote sites
	 *
	 * @return bool
	 */
	private function is_gzip_supported() {
		$supported_on_source = function_exists( 'gzcompress' );
		$supported_on_target = $this->get_remote_data( 'gzip_supported' );
		return $supported_on_source && $supported_on_target;
	}

	/**
	 * Get maximum amount of data allowed for POST request
	 *
	 * @return int
	 */
	private function get_post_max_size() {

		// Set default post max size to 25mb.
		$post_max_size = 26214400;

		// Get lowest between default, post_max_size and upload_max_filesize.
		if ( wp_max_upload_size() ) {
			$post_max_size = min( wp_max_upload_size(), $post_max_size );
		}

		// Try to get suhosin specific ini values, if applicable.
		if ( function_exists( 'ini_get' ) ) {
			$suhosin_request_limit = wp_convert_hr_to_bytes( ini_get( 'suhosin.request.max_value_length' ) );
			$suhosin_post_limit    = wp_convert_hr_to_bytes( ini_get( 'suhosin.post.max_value_length' ) );
			if ( $suhosin_request_limit ) {
				$post_max_size = min( $suhosin_request_limit, $post_max_size );
			}
			if ( $suhosin_post_limit ) {
				$post_max_size = min( $suhosin_post_limit, $post_max_size );
			}
		}

		// Account for 1kb of bloat from HTTP headers and other factors.
		return apply_filters( 'ns_cloner_teleport_post_max_size', $post_max_size - 1024 );
	}

	/**
	 * Convert array to multipart data that can be sent as a file
	 *
	 * @param array $data Array of data to encode.
	 * @return string|mixed
	 */
	private function array_to_multipart( $data ) {
		if ( ! $data || ! is_array( $data ) ) {
			return $data;
		}

		$result = '';
		foreach ( $data as $key => $value ) {
			$result .= '--' . $this->multipart_boundary . "\r\n" . sprintf( 'Content-Disposition: form-data; name="%s"', $key );
			if ( 'fragment' === $key ) {
				if ( isset( $data['gzipped'] ) && '1' === $data['gzipped'] ) {
					$result .= "; filename=\"fragment.txt.gz\"\r\nContent-Type: application/x-gzip";
				} else {
					$result .= "; filename=\"fragment.txt\"\r\nContent-Type: text/plain;";
				}
			} else {
				$result .= "\r\nContent-Type: text/plain; charset=" . get_option( 'blog_charset' );
			}
			$result .= "\r\n\r\n" . $value . "\r\n";
		}
		$result .= '--' . $this->multipart_boundary . "--\r\n";

		return $result;
	}

	/**
	 * Retrieve the data from a transmitted fragment file (containing sql or an upload file).
	 *
	 * @param bool $gzipped Whether fragment is gzipped or not.
	 * @return string
	 */
	private function get_fragment( $gzipped = false ) {
		$tmp_file_path = wp_tempnam( $gzipped ? 'fragment.txt.gz' : 'fragment.txt' );
		if ( ! isset( $_FILES['fragment']['tmp_name'] ) || ! move_uploaded_file( $_FILES['fragment']['tmp_name'], $tmp_file_path ) ) {
			$this->error = __( 'Could not upload the data to the remote server.', 'ns-cloner' );
			return '';
		} elseif ( ! file_get_contents( $tmp_file_path ) ) {
			$this->error = __( 'Could not read the data uploaded to the remote server.', 'ns-cloner' );
			unlink( $tmp_file_path );
			return '';
		} else {
			$data = file_get_contents( $tmp_file_path );
			unlink( $tmp_file_path );
			return $data;
		}
	}

}
